function cds_new = update_model_x0_v2(cds, discounts_IMM, CDX, start_date_num, end_date_num, maturity)
% --------------------------------------------------------------------------------------------------
% Update the model implied initial default intensity for a CDS (between start_date and end_date).
% Exploit that mispricing error roughly a linear function of x0.
% --------------------------------------------------------------------------------------------------
% cds               ... structure containing credit default swap time series (see 'all_steps_in_a_row.m')
% discounts_IMM     ... structure with discount curves matching IMM dates
% CDX               ... credit index structure which this CDX is part of (because need y0 & dynamics of common factor)
% start_date_num    ... datenum of start date
% end_date_num      ... datenum of end date
% discounts_cds     ... structure with discount curves (quarterly horizons)
% maturity          ... which maturity of cds.T should be used
% --------------------------------------------------------------------------------------------------
% sample call: update_model_x0_v2(cdx_mor.portfolio(1), discounts_IMM, cdx_mor, cdx_mor.dates{1}(1), cdx_mor.dates{1}(end))
%{
grid = 0.7:0.001:0.9;
error = [];
for i = 1:length(grid)
    tmp = mispricing_CDS_x0(grid(i)*0.002, cds, discounts, discounts_inbetween, maturity, start_date_num, end_date_num, CDX.liq_prem_cds);
    error(i) = tmp(2);
end
plot(grid*0.002, error);
%}
% --------------------------------------------------------------------------------------------------

% By default, assume 5-yr maturity is priced perfectly
if (nargin <= 5)
    maturity = 1;
end

% Determine relevant date range
start_pos = find(cds.dates{maturity} >= start_date_num, 1, 'first');
end_pos = find(cds.dates{maturity} <= end_date_num, 1, 'last');
used_range = start_pos:end_pos;
if (isempty(used_range))
    %cds.x0 = repmat(0.001, length(cds.dates{1}), 1);
    cds_new = cds;
    return;
end
used_dates = cds.dates{maturity}(used_range);

% Extract relevant discount curves
[discounts, discounts_inbetween] = get_discounts(used_dates, discounts_IMM, cds.T{maturity}(used_range));

% Determine constraints implied by lambda_it <= X_it + ai*Y_t
[trash, used_rows] = is_member_sorted_c(used_dates, CDX.dates{1});
%lambda_it_lower = CDX.y0(used_rows) .* cds.ai(used_range) / 0.9;     % TEST: require at least 10% idiosyncratic risk 
liq_prem_cds = CDX.liq_prem_cds(used_rows,:);

% Allocate memory
if (sum(strcmp('x0', fields(cds))) == 0) | (numel(cds.x0) == 0)    
    num_dates = length(cds.dates{maturity});
    cds.x0 = repmat(1e-6, num_dates, 1);
end

% Define serch interval parameters
%lower = lambda_it_lower;
lower = zeros(length(used_range), 1);
tmp = 0.2; upper = tmp(ones(length(used_range), 1));    % faster than repmat
precision = 1e-7;
x_old = -1;
x0_new = lower;

% num_dates = length(used_dates);
% grid = (0.01:0.001:1)' * 1e-1;
% values = zeros(num_dates, length(grid));
% for i=1:length(grid)
%     values(:,i) = mispricing_CDS_x0(repmat(grid(i), num_dates, 1), cds, discounts, discounts_inbetween, maturity, ...
%                                     start_date_num, end_date_num, liq_prem_cds);
% end
% plot(grid, values);

while (max(abs(x_old - x0_new)) > precision) & (max(upper - lower) > precision)
    % Calculate derivative of pricing error wrt x0 at lower end of search interval
    x_old = x0_new;
    lower_mispricing = mispricing_CDS_x0(x0_new, cds, discounts, discounts_inbetween, maturity, start_date_num, end_date_num, liq_prem_cds, ...
                                         CDX.AJD_common_factor, CDX.y0);
    lower_mispricing2 = mispricing_CDS_x0(x0_new + precision, cds, discounts, discounts_inbetween, maturity, start_date_num, end_date_num, liq_prem_cds, ...
                                          CDX.AJD_common_factor, CDX.y0);
    derivative = (lower_mispricing2 - lower_mispricing) / precision;
    
    % Compute optimal value of x0 implied by this derivative
    x0_new = x0_new - lower_mispricing ./ derivative;

    % Make sure that algorithm doesn't get stuck at EXACTLY the minimum
    lower(derivative < 0) = x_old(derivative < 0) + precision/2;
    upper(derivative > 0) = x_old(derivative > 0) - precision/2;
    
    % Force new value to be inside search interval
    x0_new = max(lower, min(x0_new, upper));
    %lower = x0_new;
end

% Return CDS with updated model price
cds_new = cds;
cds_new.x0(used_range) = x0_new;
